// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import com.vasco.digipass.sdk.utils.notification.server.exceptions.NotificationSDKServerException;
import com.vasco.digipass.sdk.utils.notification.server.*;

import java.util.logging.Logger;

/**
 * Test class for the NotificationSDKServer. It demonstrates how to send a notification to Android
 * and iOS devices using the NotificationSDKServer.
 */
public class NotificationSDKServerSample {
    // TODO: Paste here the VASCO notification identifier of your device,
    // retrieved from your client sample integration after the notification service registration.
    // Ex:
    // 01000001010200CC57504129317248645261724C51574C4E56396B6D6361564A756C456263327667586D70755650417454745462467663666A6D756B424C6463564266665068564B75504C76534B74386B516A37306B2D7A715748494832762D6C6968574B31326235776A4B674E4A55694C506343789E3852734D2D64536165326F37507262335048562D48736C594F6A564A6C3264694E774E4A626B79784C6F6A62676D4B654845784E5F61245E7947543443584C7744734A48123C5F597056504C3850726A7042756E5165571348EF5691A2
    private static final String ANDROID_VASCO_NOTIFICATION_IDENTIFIER =
            "<VASCO notification identifier>";

    // TODO: Paste here the path to the Firebase service account JSON configuration file to be used
    // during the communication with the server.
    // Ex: "/path/to/firebase-service-account.json"
    private static final String ANDROID_FIREBASE_SERVICE_ACCOUNT_JSON = "<path to JSON file>";

    // TODO: Set to true if you want the android notification to be heads-up.
    private static final boolean ANDROID_HEADS_UP = true;

    // TODO: Set to true if you want the notification to be sent in dry-run mode.
    private static final boolean ANDROID_DRY_RUN = false;

    // TODO: Paste here the VASCO notification identifier of your device, retrieved from your client
    // sample integration after the notification service registration.
    // Ex:
    // 01000001020100473366623931383936203431666632343663206163356562366437203735316534396635206636632131383835203736373364303646206562663435646530206233373137306230
    private static final String IOS_VASCO_NOTIFICATION_IDENTIFIER =
            "<VASCO notification identifier>";

    // TODO: Set to true if you want to use the sandbox gateway, set to false if not.
    private static final boolean IOS_USE_DEVELOPMENT_SERVER = true;

    // TODO: Paste here the path of your P12 file.
    // Ex: ./resources/apns.p12
    private static final String IOS_P12_FILE_PATH = "<path to file>";

    // TODO: Paste here the password of your P12 file.
    private static final String IOS_P12_PASSWORD = "<password>";

    // TODO: Paste here the path of your P8 file.
    // Ex: ./resources/apns.p8
    private static final String IOS_P8_FILE_PATH = "<path to file>";

    // TODO: Paste here the key ID.
    // Ex: 8Z9JNM2MG8
    private static final String IOS_KEY_ID = "<key ID>";

    // TODO: Paste here the team ID.
    // Ex: PS856PAVJ3
    private static final String IOS_TEAM_ID = "<team ID>";

    // TODO: Paste here the targeted bundle ID.
    // Ex: com.mycompany.myapp
    private static final String IOS_BUNDLE_ID = "<bundle ID>";

    private static final Logger logger =
            Logger.getLogger(NotificationSDKServerSample.class.getName());

    public static void main(String[] args) {
        NotificationSDKServer notificationServer = null;
        try {
            // Sets the credentials
            NotificationSDKServerCredentials credentials = new NotificationSDKServerCredentials();
            credentials.setAndroidFirebaseServiceAccountJson(ANDROID_FIREBASE_SERVICE_ACCOUNT_JSON);
            // token-based authentication
            credentials.setiOSPlatformCredentials(
                    IOS_USE_DEVELOPMENT_SERVER, IOS_P8_FILE_PATH, IOS_KEY_ID, IOS_TEAM_ID);
            // certificate-based authentication
            // credentials.setiOSPlatformCredentials(
            //        IOS_USE_DEVELOPMENT_SERVER, IOS_P12_FILE_PATH, IOS_P12_PASSWORD);

            // Creates the settings for the server
            NotificationSDKServerSettings settings = new NotificationSDKServerSettings();
            settings.setThreadMaxNumber(20);
            settings.setConnectionTimeout(20000);
            settings.setTerminateTimeout(60000);

            // Creates callbacks for success/error when sending a notification
            NotificationSampleCallback callbacks = new NotificationSampleCallback();

            // Creates the NotificationSDKServer object
            notificationServer = new NotificationSDKServer(credentials, settings, callbacks);

            // Build a notification that will be sent to a device
            String notificationContent = "CONTENT of the notification";
            String notificationTitle = "TITLE of the notification";
            String notificationSubject = "SUBJECT of the notification";
            Notification notification =
                    new Notification(notificationTitle, notificationSubject, notificationContent);

            // Add custom Attributes (key and value are customizable by the user)
            notification.addCustomAttribute("customAttributeKey", "customAttributeValue");

            // iOS only, sets the targeted app bundle ID
            notification.setiOSBundleId(IOS_BUNDLE_ID);

            // Android only, sets notification heads-up
            notification.setAndroidHeadsUp(ANDROID_HEADS_UP);

            // Android only, sets dry-run option
            notification.setAndroidDryRun(ANDROID_DRY_RUN);

            // Give a unique Id to the notification (allows notifications override for example)
            notification.setIdentifier(12345);

            // Uncomment to set the time to live of the notification
            // long currentTime = new Date().getTime();
            // Date oneMinuteLaterDate = new Date(currentTime + TimeUnit.MINUTES.toMillis(1));
            // notification.setTimeToLive(oneMinuteLaterDate);

            // Sends the notification to the Android device
            notificationServer.sendNotification(
                    ANDROID_VASCO_NOTIFICATION_IDENTIFIER, notification);

            // Sends the notification to the iOS device
            notificationServer.sendNotification(IOS_VASCO_NOTIFICATION_IDENTIFIER, notification);

            // Waits for the queues to drain (without stopping)
            // notificationServer.awaitSendTermination();

        } catch (NotificationSDKServerException e) {
            handleNotificationSDKServerException(e);
        } finally {
            try {
                if (notificationServer != null) {
                    // Stops and waits for the queues to drain
                    notificationServer.awaitSendTerminationAndFinish();
                }
            } catch (InterruptedException e) {
                logger.severe(() -> "InterruptedException: " + e.getMessage());
                Thread.currentThread().interrupt();
            } catch (NotificationSDKServerException e) {
                handleNotificationSDKServerException(e);
            }
        }
    }

    private static void handleNotificationSDKServerException(NotificationSDKServerException e) {
        logger.severe(() -> "NotificationSDKServerException ErrorCode: " + e.getErrorCode());

        if (e.getCause() != null && e.getCause().getMessage() != null) {
            logger.severe(
                    () ->
                            "NotificationSDKServerException Root Message: "
                                    + e.getCause().getMessage());
        }
    }
}

/** Callback class for error management while sending notifications. */
class NotificationSampleCallback implements NotificationSDKServerCallbacks {

    private static final Logger logger =
            Logger.getLogger(NotificationSampleCallback.class.getName());

    @Override
    public void onSendNotificationError(
            int trackingIdentifier,
            String vascoNotificationIdentifier,
            NotificationSDKServerException e) {
        try {
            logger.severe(
                    () ->
                            "["
                                    + trackingIdentifier
                                    + "] vascoNotificationIdentifier: "
                                    + vascoNotificationIdentifier);
            if (vascoNotificationIdentifier != null) {
                final String platformName =
                        String.valueOf(
                                NotificationSDKServer.getPlatformName(vascoNotificationIdentifier));
                logger.severe(() -> "[" + trackingIdentifier + "] platformName: " + platformName);
            }
            logger.severe(
                    () ->
                            "["
                                    + trackingIdentifier
                                    + "] NotificationSDKServerException ErrorCode: "
                                    + e.getErrorCode());
            logger.info(() -> "[" + trackingIdentifier + "] " + e.getCause().getMessage());
        } catch (NotificationSDKServerException e1) {
            e1.printStackTrace();
        }
    }

    @Override
    public void onSendNotificationSuccess(
            int trackingIdentifier, String vascoNotificationIdentifier) {
        try {
            logger.info(
                    () ->
                            "["
                                    + trackingIdentifier
                                    + "] vascoNotificationIdentifier: "
                                    + vascoNotificationIdentifier);
            if (vascoNotificationIdentifier != null) {
                final String platformName =
                        String.valueOf(
                                NotificationSDKServer.getPlatformName(vascoNotificationIdentifier));
                logger.info(() -> "[" + trackingIdentifier + "] platformName: " + platformName);
            }
            logger.info(() -> "[" + trackingIdentifier + "] Notification success.");
        } catch (NotificationSDKServerException e1) {
            e1.printStackTrace();
        }
    }
}
